#!/bin/bash

# Script to install and configure OfficeIRC Server 3 as a systemd service
# (C) Copyright 2025-2026 Office IRC Inc.

OIRC_INSTALL_DIR="/opt/oirc"
OIRC_SERVICE_FILE="oirc.service"
OIRC_FILE="OfficeIRC.dll"
SYSTEMD_SERVICE_FILE="/etc/systemd/system/oirc.service"
dotnet_version="10.0"
FONT_SOURCE_DIR="./Karla" 
FONT_TARGET_DIR="/usr/share/fonts/karla"

if [[ $EUID -ne 0 ]]; then
?? echo "This script must be run as root (using sudo)."
?? exit 1
fi

if [[ ! -f "$OIRC_FILE" ]]; then
?? ?? echo "Application files are missing. Please check the archive contents has been fully extracted."
?? ?? exit 1
fi

echo "Installing .NET $dotnet_version Runtime..."
apt-get update
apt-get install -y aspnetcore-runtime-$dotnet_version

if [[ $? -ne 0 ]]; then
?? ?? echo "Failed to install .NET $dotnet_version Runtime. Exiting."
?? ?? exit 1
fi

echo "--- Installing Karla Font ---"

if [ ! -d "$FONT_SOURCE_DIR" ]; then
    echo "Warning: Font source directory '$FONT_SOURCE_DIR' not found. Skipping font installation."
else
    mkdir -p "$FONT_TARGET_DIR" 
    sudo cp -r "$FONT_SOURCE_DIR"/* "$FONT_TARGET_DIR"/
    sudo fc-cache -f -v > /dev/null 2>&1 
    echo "Karla font installation complete."
fi

echo "Installing OfficeIRC Server 3 to $OIRC_INSTALL_DIR..."
mkdir -p "$OIRC_INSTALL_DIR"
sudo cp -r . "$OIRC_INSTALL_DIR"
sudo rm -rf "$OIRC_INSTALL_DIR/Karla"

if [[ ! -f "$OIRC_INSTALL_DIR/$OIRC_SERVICE_FILE" ]]; then
?? ?? echo "oirc.service file not found in the application directory. Please check."
?? ?? exit 1
fi

echo "Configuring and installing systemd service..."
cp "$OIRC_INSTALL_DIR/$OIRC_SERVICE_FILE" "$SYSTEMD_SERVICE_FILE"

systemctl daemon-reload
systemctl enable oirc.service
systemctl start oirc.service

if [[ $? -eq 0 ]]; then
?? ?? echo "OfficeIRC Server 3 installed and service started successfully."
else
?? ?? echo "OfficeIRC Server 3 installation finished. However, there was a problem starting the service. Check the logs."
fi

echo "Checking service status:"
systemctl status oirc.service
echo ""
echo "You can view detailed logs with: journalctl -u oirc.service"

exit 0
